#!/bin/bash
# NPM SSL 인증서 갱신 모니터링 및 텔레그램 알림

# 텔레그램 설정
TELEGRAM_BOT_TOKEN="YOUR_BOT_TOKEN_HERE"
TELEGRAM_CHAT_ID="YOUR_CHAT_ID_HERE"

# NPM 컨테이너 이름
NPM_CONTAINER="npm"

# 로그 파일 경로
LOG_FILE="/var/log/npm_ssl_monitor.log"
LAST_CHECK_FILE="/tmp/npm_last_check"

# 사용법 출력
usage() {
    echo "사용법: $0 [옵션]"
    echo ""
    echo "옵션:"
    echo "  -m, --monitor     로그 모니터링 모드 (기본)"
    echo "  -s, --status      인증서 상태 확인 및 전송"
    echo "  -r, --renew       수동 인증서 갱신 시도"
    echo "  -h, --help        도움말 표시"
    echo ""
    echo "예시:"
    echo "  $0                # 로그 모니터링 (cron용)"
    echo "  $0 -s             # 현재 인증서 상태 확인"
    echo "  $0 -r             # 수동 갱신 시도"
    exit 1
}

# 텔레그램 메시지 전송 함수 (마크다운 안전 처리)
send_telegram() {
    local message="$1"
    # 마크다운 특수문자 이스케이프 처리
    message=$(echo "$message" | sed 's/\./\\./g' | sed 's/-/\\-/g' | sed 's/(/\\(/g' | sed 's/)/\\)/g')
    
    curl -s -X POST "https://api.telegram.org/bot${TELEGRAM_BOT_TOKEN}/sendMessage" \
        -d chat_id="${TELEGRAM_CHAT_ID}" \
        -d text="${message}" \
        -d parse_mode="MarkdownV2"
}

# 안전한 텔레그램 메시지 전송 (HTML 모드 + 실제 개행)
send_telegram_safe() {
    local message="$1"
    # HTML 태그로 변환 (br 태그 제거, 실제 개행문자 사용)
    message=$(printf "%b" "$message" | \
        sed 's/\*\*\([^*]*\)\*\*/<b>\1<\/b>/g' | \
        sed 's/\*\([^*]*\)\*/<i>\1<\/i>/g' | \
        sed 's/`\([^`]*\)`/<code>\1<\/code>/g' | \
        sed 's/```\([^`]*\)```/<pre>\1<\/pre>/g')
    
    # curl로 전송 시 개행문자 유지
    curl -s -X POST "https://api.telegram.org/bot${TELEGRAM_BOT_TOKEN}/sendMessage" \
        -d chat_id="${TELEGRAM_CHAT_ID}" \
        -d text="$message" \
        -d parse_mode="HTML"
}

# 인증서 정보 확인 함수 (호스트 경로 사용)
check_certificate_info() {
    local cert_info=""
    local host_letsencrypt_path="/volume1/docker/npm/letsencrypt"
    
    # 호스트의 archive 디렉토리에서 NPM 인증서 확인
    if [ -d "$host_letsencrypt_path/archive" ]; then
        for cert_dir in "$host_letsencrypt_path/archive"/npm-*; do
            if [ -d "$cert_dir" ]; then
                # 최신 fullchain 파일 찾기
                local latest_cert=$(ls -t "$cert_dir"/fullchain*.pem 2>/dev/null | head -1)
                
                if [ -f "$latest_cert" ]; then
                    # 인증서 정보 추출
                    local cert_info_raw=$(openssl x509 -in "$latest_cert" -text -noout 2>/dev/null)
                    
                    if [ ! -z "$cert_info_raw" ]; then
                        # 도메인 정보
                        local domains=$(echo "$cert_info_raw" | grep -A1 "Subject Alternative Name" | tail -1 | sed 's/DNS://g' | sed 's/,/\n/g' | head -5)
                        
                        # 만료일 확인
                        local expiry_date=$(openssl x509 -in "$latest_cert" -enddate -noout 2>/dev/null | cut -d= -f2)
                        
                        if [ ! -z "$expiry_date" ]; then
                            # 만료일까지 남은 일수 계산
                            local expiry_epoch=$(date -d "$expiry_date" +%s 2>/dev/null)
                            local current_epoch=$(date +%s)
                            local days_remaining=$(( (expiry_epoch - current_epoch) / 86400 ))
                            
                            # 갱신 권장 일수 (30일 전)
                            local renewal_threshold=30
                            local days_until_renewal=$((days_remaining - renewal_threshold))
                            
                            # 인증서 파일명에서 번호 추출
                            local cert_number=$(basename "$latest_cert" | grep -o '[0-9]\+')
                            
                            # printf로 실제 개행문자를 포함한 문자열 생성
                            cert_info+="\n📜 <b>인증서</b>: $(basename "$cert_dir") (v$cert_number)\n"
                            cert_info+="🌐 <b>도메인</b>: $(echo "$domains" | head -1 | xargs | sed 's/^[[:space:]]*//')\n"
                            
                            if [ $(echo "$domains" | wc -l) -gt 1 ]; then
                                cert_info+="   └ 와일드카드 포함 $(echo "$domains" | wc -l)개 도메인\n"
                            fi
                            
                            cert_info+="📅 <b>만료일</b>: <code>$(date -d "$expiry_date" '+%Y-%m-%d %H:%M')</code>\n"
                            cert_info+="⏰ <b>남은 일수</b>: "
                            
                            if [ $days_remaining -le 7 ]; then
                                cert_info+="🚨 <b>$days_remaining일</b> (위험)\n"
                            elif [ $days_remaining -le 30 ]; then
                                cert_info+="⚠️ <b>$days_remaining일</b> (갱신 권장)\n"
                            else
                                cert_info+="✅ <b>$days_remaining일</b> (안전)\n"
                            fi
                            
                            if [ $days_until_renewal -le 0 ]; then
                                cert_info+="🔄 <b>갱신 시기</b>: 지금 갱신 가능\n"
                            else
                                cert_info+="🔄 <b>갱신 시기</b>: $days_until_renewal일 후\n"
                            fi
                            
                            # 인증서 파일 경로 추가 (디버그용)
                            cert_info+="📂 <b>파일</b>: <code>$(basename "$latest_cert")</code>\n"
                            cert_info+="\n"
                        fi
                    fi
                fi
            fi
        done
    else
        cert_info="\n❌ 인증서 디렉토리를 찾을 수 없습니다: $host_letsencrypt_path/archive\n"
    fi
    
    echo -e "$cert_info"
}

# 인증서 상태 확인 및 전송 (실제 개행문자 사용)
send_certificate_status() {
    local current_time=$(date '+%Y-%m-%d %H:%M:%S')
    local cert_info=$(check_certificate_info)
    
    # printf를 사용하여 실제 개행문자 생성
    local message=$(printf "📋 <b>NPM SSL 인증서 상태 보고</b>\n\n📅 <b>확인 시간</b>: <code>%s</code>\n🏠 <b>서버</b>: NPM (192.168.1.158)\n\n" "$current_time")
    
    if [ ! -z "$cert_info" ]; then
        # cert_info를 HTML 형식으로 변환 (실제 개행문자 포함)
        local html_cert_info=$(printf "%s" "$cert_info" | \
            sed 's/\*\*\([^*]*\)\*\*/<b>\1<\/b>/g' | \
            sed 's/`\([^`]*\)`/<code>\1<\/code>/g')
        message+="$html_cert_info"
    else
        message+="❌ 인증서 정보를 찾을 수 없습니다.\n"
    fi
    
    message+="\n💡 <b>참고</b>: 만료 30일 전부터 자동 갱신됩니다."
    
    send_telegram_safe "$message"
    echo "[$current_time] 인증서 상태 보고 전송됨" >> "$LOG_FILE"
}

# 수동 갱신 시도 (안전한 메시지 형식)
manual_renewal() {
    local current_time=$(date '+%Y-%m-%d %H:%M:%S')
    
    # 갱신 전 상태 확인
    local before_info=$(check_certificate_info)
    
    # 수동 갱신 실행
    local renewal_output=$(docker exec "$NPM_CONTAINER" certbot renew --force-renewal 2>&1)
    local renewal_exit_code=$?
    
    local message="🔄 <b>NPM SSL 수동 갱신 시도</b>\n\n"
    message+="📅 <b>시간</b>: <code>$current_time</code>\n"
    message+="🏠 <b>서버</b>: NPM (192.168.1.158)\n\n"
    
    if [ $renewal_exit_code -eq 0 ]; then
        # 갱신 후 상태 확인
        sleep 5
        local after_info=$(check_certificate_info)
        
        message+="✅ <b>결과</b>: 갱신 성공\n\n"
        message+="📋 <b>갱신 후 인증서 정보</b>:\n"
        
        # HTML 형식으로 변환
        local html_after_info=$(echo "$after_info" | \
            sed 's/\*\*\([^*]*\)\*\*/<b>\1<\/b>/g' | \
            sed 's/`\([^`]*\)`/<code>\1<\/code>/g')
        message+="$html_after_info"
    else
        message+="❌ <b>결과</b>: 갱신 실패\n\n"
        message+="📝 <b>오류 내용</b>:\n"
        message+="<pre>$(echo "$renewal_output" | tail -10)</pre>\n\n"
        message+="📋 <b>현재 인증서 정보</b>:\n"
        
        # HTML 형식으로 변환
        local html_before_info=$(echo "$before_info" | \
            sed 's/\*\*\([^*]*\)\*\*/<b>\1<\/b>/g' | \
            sed 's/`\([^`]*\)`/<code>\1<\/code>/g')
        message+="$html_before_info"
    fi
    
    send_telegram_safe "$message"
    echo "[$current_time] 수동 갱신 시도 결과 전송됨 (종료코드: $renewal_exit_code)" >> "$LOG_FILE"
}

# 로그 모니터링 함수 (기존 기능) - 안전한 메시지
monitor_logs() {
    # 현재 시간
    local current_time=$(date '+%Y-%m-%d %H:%M:%S')

    # 마지막 체크 시간 읽기
    if [ -f "$LAST_CHECK_FILE" ]; then
        local last_check=$(cat "$LAST_CHECK_FILE")
    else
        local last_check="1970-01-01 00:00:00"
    fi

    # NPM 로그에서 SSL 갱신 관련 내용 확인
    local renewal_logs=$(docker logs "$NPM_CONTAINER" --since "$last_check" 2>/dev/null | grep -i "renew\|ssl.*cert\|certificate.*success\|certificate.*fail")

    if [ ! -z "$renewal_logs" ]; then
        # 성공 메시지 확인
        local success_count=$(echo "$renewal_logs" | grep -i "success\|completed.*renew\|certificate.*renewed" | wc -l)
        
        # 실패 메시지 확인  
        local fail_count=$(echo "$renewal_logs" | grep -i "fail\|error\|invalid" | wc -l)
        
        if [ $success_count -gt 0 ]; then
            local message="✅ <b>NPM SSL 인증서 갱신 성공</b>\n\n"
            message+="📅 <b>시간</b>: <code>$current_time</code>\n"
            message+="🔄 <b>갱신된 인증서 수</b>: $success_count\n"
            message+="🏠 <b>서버</b>: NPM (192.168.1.158)\n\n"
            
            # 갱신 후 인증서 정보 추가
            local cert_info=$(check_certificate_info)
            if [ ! -z "$cert_info" ]; then
                message+="📋 <b>갱신된 인증서 정보</b>:\n"
                # HTML 형식으로 변환
                local html_cert_info=$(echo "$cert_info" | \
                    sed 's/\*\*\([^*]*\)\*\*/<b>\1<\/b>/g' | \
                    sed 's/`\([^`]*\)`/<code>\1<\/code>/g')
                message+="$html_cert_info"
            fi
            
            message+="✨ 모든 SSL 인증서가 성공적으로 갱신되었습니다."
            
            send_telegram_safe "$message"
            echo "[$current_time] SSL 갱신 성공 알림 전송됨" >> "$LOG_FILE"
        fi
        
        if [ $fail_count -gt 0 ]; then
            local message="❌ <b>NPM SSL 인증서 갱신 실패</b>\n\n"
            message+="📅 <b>시간</b>: <code>$current_time</code>\n"
            message+="⚠️ <b>실패한 인증서 수</b>: $fail_count\n"
            message+="🏠 <b>서버</b>: NPM (192.168.1.158)\n\n"
            message+="SSL 인증서 갱신에 실패했습니다. 확인이 필요합니다.\n\n"
            message+="<b>로그 확인 명령어</b>:\n"
            message+="<code>docker logs npm | tail -50</code>"
            
            send_telegram_safe "$message"
            echo "[$current_time] SSL 갱신 실패 알림 전송됨" >> "$LOG_FILE"
        fi
    fi

    # 현재 시간을 마지막 체크 시간으로 저장
    echo "$current_time" > "$LAST_CHECK_FILE"
}

# 메인 실행 로직
case "${1:-}" in
    -s|--status)
        echo "인증서 상태를 확인하고 텔레그램으로 전송합니다..."
        send_certificate_status
        ;;
    -r|--renew)
        echo "수동으로 인증서 갱신을 시도합니다..."
        manual_renewal
        ;;
    -h|--help)
        usage
        ;;
    -m|--monitor|"")
        # 기본 모니터링 모드 (cron용)
        monitor_logs
        ;;
    *)
        echo "잘못된 옵션: $1"
        usage
        ;;
esac